<?php

declare(strict_types=1);

namespace DeMarketplace\Auth\Tests;

use DeMarketplace\Auth\Exceptions\InvalidCredentialsException;
use DeMarketplace\Auth\Exceptions\TokenMissingException;
use DeMarketplace\Auth\Http\Middleware\Authenticate;
use DeMarketplace\Auth\Services\Auth\AuthDriverInterface;

class AuthenticateMiddlewareTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_authenticates_user(): void
    {
        $request = new class {
            /**
             * @return string
             */
            public function ok()
            {
                return 'OK';
            }
        };

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('hasToken')->willReturn(true);
        $authDriver->expects($this->once())->method('authenticateWithToken');

        $middleware = new Authenticate($authDriver);

        $result = $middleware->handle($request, function ($request) {
            return $request->ok();
        });

        $this->assertEquals('OK', $result);
    }

    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_throws_exception_when_token_is_missing()
    {
        $request = new class {};

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('hasToken')->willReturn(false);

        $this->expectException(TokenMissingException::class);
        $this->expectExceptionMessage('Token missing.');

        $middleware = new Authenticate($authDriver);
        $middleware->handle($request, function () {});
    }

    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_throws_exception_when_cannot_authenticate_user()
    {
        $request = new class {};

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())
            ->method('hasToken')
            ->willReturn(true);
        $authDriver->expects($this->once())
            ->method('authenticateWithToken')
            ->willThrowException(new InvalidCredentialsException());

        $this->expectException(InvalidCredentialsException::class);
        $this->expectExceptionMessage('Invalid credentials.');

        $middleware = new Authenticate($authDriver);
        $middleware->handle($request, function () {});

    }
}
