<?php

declare(strict_types=1);

namespace DeMarketplace\Auth\Tests;

use DeMarketplace\Auth\Exceptions\AuthException;
use DeMarketplace\Auth\Exceptions\TokenMissingException;
use DeMarketplace\Auth\Http\Middleware\RefreshToken;
use DeMarketplace\Auth\Services\Auth\AuthDriverInterface;

class RefreshTokenMiddlewareTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_refreshes_the_token(): void
    {
        $request = new class {
            /**
             * @return string
             */
            public function ok(): string
            {
                return 'OK';
            }
        };

        $response = new class {
            /**
             * @var array
             */
            private $data = [];

            /**
             * @param string $key
             * @param string $value
             * @return mixed
             */
            public function header(string $key, string $value)
            {
                $this->data[$key] = $value;

                return $this;
            }

            /**
             * @return array
             */
            public function getData(): array
            {
                return $this->data;
            }
        };

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('hasToken')->willReturn(true);
        $authDriver->expects($this->once())->method('refreshToken')->willReturn('somejwttoken');

        $middleware = new RefreshToken($authDriver);

        $result = $middleware->handle($request, function ($request) use ($response) {
            return $response;
        });

        $this->assertEquals(['Authorization' => 'Bearer somejwttoken'], $result->getData());
    }

    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_throws_exception_when_token_is_missing()
    {
        $request = new class {};

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('hasToken')->willReturn(false);

        $this->expectException(TokenMissingException::class);
        $this->expectExceptionMessage('Token missing.');

        $middleware = new RefreshToken($authDriver);
        $middleware->handle($request, function () {});
    }

    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_throws_exception_when_cannot_refresh_token()
    {
        $request = new class {};

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())
            ->method('hasToken')
            ->willReturn(true);
        $authDriver->expects($this->once())
            ->method('refreshToken')
            ->willThrowException(new AuthException('Cannot refresh token.'));

        $this->expectException(AuthException::class);
        $this->expectExceptionMessage('Cannot refresh token.');

        $middleware = new RefreshToken($authDriver);
        $middleware->handle($request, function () {});
    }
}
