<?php

declare(strict_types=1);

namespace DeMarketplace\Auth;

use DeMarketplace\Auth\Exceptions\Converter;
use DeMarketplace\Auth\Http\Middleware\Authenticate;
use DeMarketplace\Auth\Http\Middleware\RefreshToken;
use DeMarketplace\Auth\Http\Middleware\SetToken;
use DeMarketplace\Auth\Services\Auth\AuthDriverInterface;
use DeMarketplace\Auth\Services\Auth\AuthServiceManager;
use DeMarketplace\Auth\Services\AuthenticationService;
use DeMarketplace\Auth\Services\AuthenticationServiceInterface;
use DeMarketplace\Auth\Services\Password\ResetPasswordManager;
use DeMarketplace\Auth\Services\PasswordService;
use DeMarketplace\Auth\Services\PasswordServiceInterface;
use DeMarketplace\Auth\Services\RegistrationService;
use DeMarketplace\Auth\Services\RegistrationServiceInterface;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Support\ServiceProvider;

class AuthServiceProvider extends ServiceProvider
{
    /**
     * @return void
     */
    public function register(): void
    {
        $this->registerAuthenticationService();
        $this->registerRegistrationService();
        $this->registerPasswordService();
    }

    /**
     * @return void
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    public function boot(): void
    {
        $this->aliasMiddlewares();
        $this->registerExceptionConverter();
    }

    /**
     * @return void
     */
    private function registerAuthenticationService(): void
    {
        $this->app->bind(AuthDriverInterface::class, function () {
            return (new AuthServiceManager($this->app))->driver($this->app['config']['auth.guards.api.driver']);
        });

        $this->app->bind(AuthenticationServiceInterface::class, function () {
            return new AuthenticationService($this->app->make(AuthDriverInterface::class));
        });
    }

    /**
     * @return void
     */
    private function registerRegistrationService(): void
    {
        $this->app->bind(RegistrationServiceInterface::class, function () {
            return new RegistrationService(
                $this->app['config'],
                $this->app[Dispatcher::class],
                $this->app['hash']
            );
        });
    }

    /**
     * @return void
     */
    private function registerPasswordService(): void
    {
        $this->app->bind(PasswordServiceInterface::class, function () {
            return new PasswordService(
                new ResetPasswordManager($this->app),
                $this->app[Dispatcher::class],
                $this->app['hash']
            );
        });
    }

    /**
     * @return void
     */
    private function aliasMiddlewares(): void
    {
        /** @var \Illuminate\Routing\Router $router */
        $router = $this->app['router'];
        $router->aliasMiddleware('demarketplace.authenticate', Authenticate::class);
        $router->aliasMiddleware('demarketplace.token.refresh', RefreshToken::class);
        $router->aliasMiddleware('demarketplace.token.set', SetToken::class);
    }

    /**
     * @return void
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    private function registerExceptionConverter(): void
    {
        $exceptionHandler = $this->app[ExceptionHandler::class];

        if (method_exists($exceptionHandler, 'registerCustomConverter')) {
            $exceptionHandler->registerCustomConverter(
                new Converter($this->app->make(AuthDriverInterface::class))
            );
        }
    }
}
