<?php

declare(strict_types=1);

namespace DeMarketplace\Auth\Services;

use DeMarketplace\Auth\Exceptions\InvalidCredentialsException;
use DeMarketplace\Auth\Exceptions\InvalidResetPasswordTokenException;
use DeMarketplace\Auth\Services\Password\ResetPasswordDriverInterface;
use DeMarketplace\Auth\Services\Password\ResetPasswordManager;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Contracts\Hashing\Hasher;
use Illuminate\Database\Eloquent\Model;

class PasswordService implements PasswordServiceInterface
{
    /**
     * @var ResetPasswordManager
     */
    private ResetPasswordManager $manager;

    /**
     * @var Dispatcher
     */
    private Dispatcher $dispatcher;

    /**
     * @var Hasher
     */
    private Hasher $hasher;

    /**
     * @var string|null
     */
    private ?string $resetPasswordMethod = null;

    /**
     * @param ResetPasswordManager $manager
     * @param Dispatcher $dispatcher
     * @param Hasher $hasher
     */
    public function __construct(ResetPasswordManager $manager, Dispatcher $dispatcher, Hasher $hasher)
    {
        $this->manager = $manager;
        $this->dispatcher = $dispatcher;
        $this->hasher = $hasher;
    }

    /**
     * @param array $credentials
     * @return void
     * @throws InvalidCredentialsException
     */
    public function requestForReset(array $credentials): void
    {
        $this->getDriver()->requestForReset($credentials);
    }

    /**
     * @param array $credentials
     * @return void
     * @throws InvalidCredentialsException
     * @throws InvalidResetPasswordTokenException
     */
    public function reset(array $credentials): void
    {
        $this->getDriver()->reset($credentials, function ($user, $password) {
            /** @var Model|Authenticatable $user */
            $user->setAttribute('password', $this->hasher->make($password));
            $user->save();

            $this->dispatcher->dispatch(new PasswordReset($user));
        });
    }

    /**
     * @param string $resetPasswordMethod
     * @return PasswordServiceInterface
     */
    public function setResetPasswordMethod(string $resetPasswordMethod): PasswordServiceInterface
    {
        $this->resetPasswordMethod = $resetPasswordMethod;

        return $this;
    }

    /**
     * @return ResetPasswordDriverInterface
     */
    public function getDriver(): ResetPasswordDriverInterface
    {
        return $this->manager->driver($this->resetPasswordMethod);
    }
}
