<?php

declare(strict_types=1);

namespace DeMarketplace\Auth\Services\Auth;

use DeMarketplace\Auth\Exceptions\AuthException;
use DeMarketplace\Auth\Exceptions\InvalidCredentialsException;
use DeMarketplace\Auth\Exceptions\TokenBlacklistedException;
use DeMarketplace\Auth\Exceptions\TokenExpiredException;
use DeMarketplace\Auth\Exceptions\TokenInvalidException;
use DeMarketplace\Auth\Exceptions\TokenNotCreatedException;
use Tymon\JWTAuth\Contracts\JWTSubject;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Exceptions\TokenBlacklistedException as JWTTokenBlacklistedException;
use Tymon\JWTAuth\Exceptions\TokenExpiredException as JWTTokenExpiredException;
use Tymon\JWTAuth\Exceptions\TokenInvalidException as JWTTokenInvalidException;
use Tymon\JWTAuth\Exceptions\UserNotDefinedException as JWTUserNotDefinedException;
use Tymon\JWTAuth\JWTAuth;

class JwtAuthDriver implements AuthDriverInterface
{
    /**
     * @var JWTAuth
     */
    private JWTAuth $jwt;

    /**
     * @param JWTAuth $jwt
     */
    public function __construct(JWTAuth $jwt)
    {
        $this->jwt = $jwt;
    }

    /**
     * @return void
     * @throws AuthException
     */
    public function authenticateWithToken(): void
    {
        try {
            $user = $this->jwt->authenticate();

            if (!$user instanceof JWTSubject) {
                throw new InvalidCredentialsException();
            }
        } catch (JWTUserNotDefinedException $e) {
            throw new InvalidCredentialsException();
        } catch (JWTTokenExpiredException $e) {
            throw new TokenExpiredException();
        } catch (JWTTokenBlacklistedException $e) {
            throw new TokenBlacklistedException();
        } catch (JWTTokenInvalidException $e) {
            throw new TokenInvalidException();
        } catch (JWTException $e) {
            throw new AuthException($e->getMessage());
        }
    }

    /**
     * @param array $credentials
     * @return void
     * @throws AuthException
     */
    public function authenticateWithCredentials(array $credentials = []): void
    {
        $attemptResult = $this->jwt->attempt($credentials);

        if (false === $attemptResult) {
            throw new InvalidCredentialsException();
        }
    }

    /**
     * @return bool
     */
    public function hasToken(): bool
    {
        return null !== $this->jwt->getToken();
    }

    /**
     * @return bool
     */
    public function isTokenValid(): bool
    {
        try {
            return $this->jwt->check();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * @return void
     * @throws AuthException
     */
    public function invalidateToken(): void
    {
        try {
            $this->jwt->invalidate();
        } catch (JWTException $e) {
            throw new AuthException($e->getMessage());
        }
    }

    /**
     * @return void
     * @throws AuthException
     */
    public function invalidateTokenForever(): void
    {
        try {
            $this->jwt->invalidate(true);
        } catch (JWTException $e) {
            throw new AuthException($e->getMessage());
        }
    }

    /**
     * @return string
     * @throws AuthException
     */
    public function refreshToken(): string
    {
        try {
            return $this->jwt->refresh();
        } catch (JWTException $e) {
            throw new AuthException($e->getMessage());
        }
    }

    /**
     * @return string
     * @throws AuthException
     */
    public function refreshTokenAndInvalidateForever(): string
    {
        try {
            return $this->jwt->refresh(true);
        } catch (JWTException $e) {
            throw new AuthException($e->getMessage());
        }
    }

    /**
     * @param JWTSubject $user
     * @return string
     * @throws TokenNotCreatedException
     */
    public function generateTokenFromUser($user): string
    {
        if (!$user instanceof JWTSubject) {
            throw new TokenNotCreatedException();
        }

        return $this->jwt->fromSubject($user);
    }

    /**
     * @param JWTSubject $user
     * @param array $customData
     * @return string
     * @throws TokenNotCreatedException
     */
    public function generateTokenFromUserWithCustomData($user, array $customData = []): string
    {
        if (!$user instanceof JWTSubject) {
            throw new TokenNotCreatedException();
        }

        return $this->jwt->claims($customData)->fromSubject($user);
    }

    /**
     * @return JWTSubject|null
     */
    public function getUser(): ?JWTSubject
    {
        return $this->jwt->user();
    }

    /**
     * @return JWTAuth
     */
    public function getAuth(): JWTAuth
    {
        return $this->jwt;
    }

    /**
     * @param string $key
     * @return mixed
     */
    public function getDataFromToken($key)
    {
        return $this->jwt->getClaim($key);
    }

    /**
     * @param string $method
     * @param array $parameters
     * @return mixed
     */
    public function __call($method, $parameters)
    {
        return call_user_func_array([$this->jwt, $method], $parameters);
    }
}
