<?php

declare(strict_types=1);

namespace DeMarketplace\Auth\Tests;

use DeMarketplace\Auth\Exceptions\TokenMissingException;
use DeMarketplace\Auth\Http\Middleware\SetToken;
use DeMarketplace\Auth\Services\Auth\AuthDriverInterface;

class SetTokenMiddlewareTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_sets_the_token(): void
    {
        $user = new class {};

        $request = new class {
            /**
             * @return string
             */
            public function ok(): string
            {
                return 'OK';
            }
        };

        $response = new class {
            /**
             * @var array
             */
            private $data = [];

            /**
             * @param string $key
             * @param string $value
             * @return mixed
             */
            public function header(string $key, string $value)
            {
                $this->data[$key] = $value;

                return $this;
            }

            /**
             * @return array
             */
            public function getData(): array
            {
                return $this->data;
            }
        };

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('getUser')->willReturn($user);
        $authDriver->expects($this->once())->method('generateTokenFromUser')->with($user)->willReturn('somejwttoken');

        $middleware = new SetToken($authDriver);

        $result = $middleware->handle($request, function ($request) use ($response) {
            return $response;
        });

        $this->assertEquals(['Authorization' => 'Bearer somejwttoken'], $result->getData());
    }

    /**
     * @test
     * @return void
     * @throws \ReflectionException
     */
    public function it_does_not_set_token_when_there_is_no_authenticated_user()
    {
        $request = new class {
            /**
             * @return string
             */
            public function ok(): string
            {
                return 'OK';
            }
        };

        $response = new class {
            /**
             * @var array
             */
            private $data = [];

            /**
             * @param string $key
             * @param string $value
             * @return mixed
             */
            public function header(string $key, string $value)
            {
                $this->data[$key] = $value;

                return $this;
            }

            /**
             * @return array
             */
            public function getData(): array
            {
                return $this->data;
            }
        };

        $authDriver = $this->createMock(AuthDriverInterface::class);
        $authDriver->expects($this->once())->method('getUser')->willReturn(null);

        $middleware = new SetToken($authDriver);

        $result = $middleware->handle($request, function ($request) use ($response) {
            return $response;
        });

        $this->assertNotContains('Authorization', $result->getData());
    }
}
