<?php

declare(strict_types=1);

namespace DeMarketplace\Payments\DirectDebit;

use DeMarketplace\Payments\DirectDebit\Http\Actions\HandleWebhookAction;
use DeMarketplace\Payments\DirectDebit\Log\Logger;
use DeMarketplace\Payments\DirectDebit\Processors\RedirectFlow\RedirectFlowProcessorInterface;
use DeMarketplace\Payments\DirectDebit\Processors\RedirectFlow\RedirectFlowProcessorManager;
use DeMarketplace\Payments\DirectDebit\Processors\Webhook\WebhookProcessorInterface;
use DeMarketplace\Payments\DirectDebit\Processors\Webhook\WebhookProcessorManager;
use GoCardlessPro\Client;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;

class DirectDebitPaymentsServiceProvider extends ServiceProvider
{
    /**
     * @return void
     */
    public function register(): void
    {
        $this->registerGoCardlessClient();

        $this->registerLogger();
        $this->registerProcessors();
        $this->registerDirectDebitService();
    }

    /**
     * @return void
     */
    public function boot(): void
    {
        $this->publishConfig();
        $this->registerRoutes();
    }

    /**
     * @return void
     */
    private function registerGoCardlessClient(): void
    {
        $this->app->bind(Client::class, function () {
            return new Client([
                'access_token' => $this->app['config']['direct-debit-payments.go_cardless.access_token'],
                'environment' => $this->app['config']['direct-debit-payments.go_cardless.environment'],
            ]);
        });
    }

    /**
     * @return void
     */
    private function registerLogger(): void
    {
        $this->app->bind(Logger::class, function () {
            return new Logger($this->app['log'], $this->app['config']['direct-debit-payments.logging']);
        });
    }

    /**
     * @return void
     */
    private function registerProcessors(): void
    {
        $this->app->bind(RedirectFlowProcessorInterface::class, function () {
            return (new RedirectFlowProcessorManager($this->app))->driver($this->app['config']['direct-debit-payments.provider']);
        });

        $this->app->bind(WebhookProcessorInterface::class, function () {
            return (new WebhookProcessorManager($this->app))->driver($this->app['config']['direct-debit-payments.provider']);
        });
    }

    /**
     * @return void
     */
    private function registerDirectDebitService(): void
    {
        $this->app->bind(DirectDebitServiceInterface::class, function () {
            return new DirectDebitService($this->app->make(RedirectFlowProcessorInterface::class));
        });
    }

    /**
     * @return void
     */
    private function publishConfig(): void
    {
        $path = realpath(__DIR__.'/../config/config.php');

        $this->publishes([$path => config_path('direct-debit-payments.php')], 'config');
        $this->mergeConfigFrom($path, 'direct-debit-payments');
    }

    /**
     * @return void
     */
    private function registerRoutes(): void
    {
        $this->app->booted(function () {
            if (!$this->app->routesAreCached()) {
                Route::prefix('api')
                    ->middleware('api')
                    ->group(function () {
                        Route::post('/webhooks/direct-debit', HandleWebhookAction::class);
                    });
            }
        });
    }
}
