<?php

declare(strict_types=1);

namespace DeMarketplace\Payments\DirectDebit\Processors\RedirectFlow;

use DeMarketplace\Payments\DirectDebit\Exceptions\RedirectFlowException;
use DeMarketplace\Payments\DirectDebit\Log\Logger;
use DeMarketplace\Payments\DirectDebit\Models\DirectDebitCustomerInterface;
use DeMarketplace\Support\HMAC;
use GoCardlessPro\Client;
use GoCardlessPro\Core\Exception\InvalidStateException;
use Illuminate\Contracts\Config\Repository as Config;

class GoCardlessRedirectFlowProcessor implements RedirectFlowProcessorInterface
{
    /**
     * @var Client
     */
    private Client $client;

    /**
     * @var Config
     */
    private Config $config;

    /**
     * @var Logger $logger
     */
    private Logger $logger;

    /**
     * @param Client $client
     * @param Config $config
     * @param Logger $logger
     */
    public function __construct(Client $client, Config $config, Logger $logger)
    {
        $this->client = $client;
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * @param DirectDebitCustomerInterface $user
     * @param string $successRedirectUrl
     * @return string
     * @throws RedirectFlowException
     */
    public function start(DirectDebitCustomerInterface $user, string $successRedirectUrl): string
    {
        try {
            $redirectFlow = $this->client->redirectFlows()->create([
                'params' => [
                    'description' => $this->config->get('direct-debit-payments.go_cardless.company_name'),
                    'session_token' => $this->getSessionToken($user),
                    'success_redirect_url' => $successRedirectUrl,
                    'prefilled_customer' => [
                        'given_name' => $user->getGivenName(),
                        'family_name' => $user->getFamilyName(),
                        'email' => $user->getEmail(),
                        'address_line1' => $user->getAddressLine(),
                        'city' => $user->getCity(),
                        'postal_code' => $user->getPostalCode(),
                    ]
                ]
            ]);

            return $redirectFlow->redirect_url;
        } catch (InvalidStateException $e) {
            $this->logger->warning(
                "Failed to start redirect flow. Message: {$e->getMessage()}",
                ['user' => $user->getIdentifier()]
            );

            throw new RedirectFlowException("Failed to start redirect flow. Message: {$e->getMessage()}");
        }
    }

    /**
     * @param DirectDebitCustomerInterface $user
     * @param string $redirectFlowId
     * @return string
     * @throws RedirectFlowException
     */
    public function complete(DirectDebitCustomerInterface $user, string $redirectFlowId): string
    {
        try {
            $redirectFlow = $this->client->redirectFlows()->complete($redirectFlowId, [
                'params' => [
                    'session_token' => $this->getSessionToken($user),
                ]
            ]);

            return $redirectFlow->links->mandate;
        } catch (InvalidStateException $e) {
            $this->logger->warning(
                "Failed to start redirect flow. Message: {$e->getMessage()}",
                ['user' => $user->getIdentifier()]
            );

            throw new RedirectFlowException("Failed to complete redirect flow. Message: {$e->getMessage()}");
        }
    }

    /**
     * @param DirectDebitCustomerInterface $user
     * @return string
     */
    private function getSessionToken(DirectDebitCustomerInterface $user): string
    {
        return (string) HMAC::create($this->config->get('app.key'), $user->getIdentifier());
    }
}
