<?php

declare(strict_types=1);

namespace DeMarketplace\Payments\DirectDebit\Processors\Webhook;

use DeMarketplace\Payments\DirectDebit\Events\IncomingMandatesWebhook;
use DeMarketplace\Payments\DirectDebit\Events\IncomingPaymentsWebhook;
use DeMarketplace\Payments\DirectDebit\Events\IncomingPayoutsWebhook;
use DeMarketplace\Payments\DirectDebit\Events\IncomingRefundsWebhook;
use DeMarketplace\Payments\DirectDebit\Events\IncomingSubscriptionsWebhook;
use DeMarketplace\Payments\DirectDebit\Log\Logger;
use GoCardlessPro\Core\Exception\InvalidSignatureException;
use GoCardlessPro\Resources\Event;
use GoCardlessPro\Webhook;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class GoCardlessWebhookProcessor implements WebhookProcessorInterface
{
    /**
     * @var Dispatcher
     */
    private Dispatcher $dispatcher;

    /**
     * @var Logger
     */
    private Logger $logger;

    /**
     * @var string
     */
    private string $webhookSecret;

    /**
     * @param Dispatcher $dispatcher
     * @param Logger $logger
     * @param string $webhookSecret
     */
    public function __construct(Dispatcher $dispatcher, Logger $logger, string $webhookSecret)
    {
        $this->dispatcher = $dispatcher;
        $this->logger = $logger;
        $this->webhookSecret = $webhookSecret;
    }

    /**
     * @see https://developer.gocardless.com/api-reference/#core-endpoints-events for events detailed descriptions.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function process(Request $request): JsonResponse
    {
        $webhookSignature = $request->header('Webhook-Signature');
        $body = $request->getContent();

        try {
            $events = Webhook::parse($body, $webhookSignature, $this->webhookSecret);

            foreach ($events as $event) {
                /** @var Event $event */

                $webhookData = [
                    'id' => $event->id,
                    'action' => $event->action,
                    'links' => (array) $event->links,
                    'details' => (array) $event->details,
                    'metadata' => (array) $event->metadata,
                    'created_at' => $event->created_at,
                ];

                switch ($event->resource_type) {
                    case 'payments':
                        $event = new IncomingPaymentsWebhook('go_cardless', $webhookData);
                        break;
                    case 'mandates':
                        $event = new IncomingMandatesWebhook('go_cardless', $webhookData);
                        break;
                    case 'payouts':
                        $event = new IncomingPayoutsWebhook('go_cardless', $webhookData);
                        break;
                    case 'subscriptions':
                        $event = new IncomingSubscriptionsWebhook('go_cardless', $webhookData);
                        break;
                    case 'refunds':
                        $event = new IncomingRefundsWebhook('go_cardless', $webhookData);
                        break;
                    default:
                        throw new \RuntimeException("Unknown GoCardless webhook resource type: {$event->resource_type}.");
                }

                $this->dispatcher->dispatch($event);
            }

            return new JsonResponse([]);
        } catch (InvalidSignatureException $e) {
            $this->logger->notice('Invalid webhook signature', ['body' => $body]);

            header('HTTP/1.1 498 Invalid Token');
            exit();
        }
    }
}
