<?php

declare(strict_types=1);

namespace DeMarketplace\Products\Models\Concerns;

use DeMarketplace\Support\MoneyFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Money\Money;

/**
 * @mixin Model
 */
trait HasPrice
{
    /**
     * @return string
     */
    protected function getHasPriceFillableAttribute(): string
    {
        return 'price';
    }

    /**
     * @param Money|int|null $amount
     * @return self
     */
    public function setPriceAttribute($amount): self
    {
        if ($amount instanceof Money) {
            $amount = $amount->getAmount();
        }

        $this->attributes[$this->getHasPriceFillableAttribute()] = $amount;

        return $this;
    }

    /**
     * @param int|null $value
     * @return Money|null
     */
    public function getPriceAttribute(?int $value): ?Money
    {
        if (null === $value) {
            return null;
        }

        return MoneyFactory::fromInteger($value);
    }

    /**
     * @return Money|null
     */
    public function getPrice(): ?Money
    {
        return $this->getAttribute($this->getHasPriceFillableAttribute());
    }

    /**
     * @return bool
     */
    public function hasPriceChanged(): bool
    {
        return (int) Arr::get($this->getAttributes(), 'price') !== (int) $this->getOriginal('price');
    }
}
