<?php

namespace DeMarketplace\Support;

use Money\Currency;
use Money\Money;

class MoneyFactory
{
    public const CURRENCY_AMOUNT_FORMAT = '{CURRENCY}{AMOUNT}';
    public const CURRENCY_SPACE_AMOUNT_FORMAT = '{CURRENCY}{SPACE}{AMOUNT}';
    public const AMOUNT_CURRENCY_FORMAT = '{AMOUNT}{CURRENCY}';
    public const AMOUNT_SPACE_CURRENCY_FORMAT = '{AMOUNT}{SPACE}{CURRENCY}';

    /**
     * @var array
     */
    private static array $isoToSymbolMap = [
        'GBP' => '£',
        'EUR' => '€',
        'USD' => '$',
    ];

    /**
     * @var string
     */
    public static string $defaultCurrency = 'GBP';

    /**
     * @var string
     */
    public static string $prettyPrintFormat = self::CURRENCY_SPACE_AMOUNT_FORMAT;

    /**
     * @param string $iso
     * @param string $symbol
     * @return void
     */
    public static function addIsoToSymbolMapping(string $iso, string $symbol): void
    {
        static::$isoToSymbolMap[$iso] = $symbol;
    }

    /**
     * @param float $amount
     * @param string|null $currency
     * @return Money
     */
    public static function fromFloat(float $amount, ?string $currency = null): Money
    {
        return new Money(round($amount * 100), new Currency($currency ?? static::$defaultCurrency));
    }

    /**
     * @param int $amount
     * @param string|null $currency
     * @return Money
     */
    public static function fromInteger(int $amount, ?string $currency = null): Money
    {
        return new Money($amount, new Currency($currency ?? static::$defaultCurrency));
    }

    /**
     * @param Money $money
     * @return float
     */
    public static function toFloat(Money $money): float
    {
        return $money->getAmount() / 100;
    }

    /**
     * @param Money $money
     * @param bool $htmlEntities
     * @return string
     */
    public static function prettyPrint(Money $money, bool $htmlEntities = false): string
    {
        $float = $money->absolute()->getAmount() / 100;
        $cents = $float - floor($float);
        $decimals = $cents > 0 ? 2 : 0;

        $currency = self::currencyToSymbol($money->getCurrency());
        $amount = number_format($float, $decimals);
        $space = $htmlEntities ? '&nbsp;' : ' ';

        $prettyPrintFormat = $money->isNegative()
            ? '-' . $space . static::$prettyPrintFormat
            : static::$prettyPrintFormat;

        return str_replace(
            ['{CURRENCY}', '{AMOUNT}', '{SPACE}'],
            [$currency, $amount, $space],
            $prettyPrintFormat
        );
    }

    /**
     * @param Currency $currency
     * @return string
     */
    public static function currencyToSymbol(Currency $currency): string
    {
        return isset(self::$isoToSymbolMap[$currency->getCode()])
            ? self::$isoToSymbolMap[$currency->getCode()]
            : $currency->getCode();
    }

    /**
     * @param Money $money
     * @return array
     */
    public static function jsonSerialize(Money $money): array
    {
        $serialized = $money->jsonSerialize();
        $serialized['formatted'] = self::prettyPrint($money);
        $serialized['float'] = self::toFloat($money);

        return $serialized;
    }
}
