<?php

declare(strict_types=1);

namespace DeMarketplace\Support\Tests;

use DeMarketplace\Support\MoneyFactory;
use Money\Currency;
use Money\Money;

class MoneyFactoryTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @test
     * @return void
     */
    public function it_instantiates_money_object(): void
    {
        $this->assertEquals('GBP', MoneyFactory::fromInteger(1000)->getCurrency()->getCode());
        $this->assertEquals('USD', MoneyFactory::fromInteger(1000, 'USD')->getCurrency()->getCode());

        $this->assertEquals('GBP', MoneyFactory::fromFloat(1000.30)->getCurrency()->getCode());
        $this->assertEquals('USD', MoneyFactory::fromFloat(1000.30, 'USD')->getCurrency()->getCode());

        MoneyFactory::$defaultCurrency = 'PLN';

        $this->assertEquals('PLN', MoneyFactory::fromInteger(1000)->getCurrency()->getCode());
        $this->assertEquals('GBP', MoneyFactory::fromInteger(1000, 'GBP')->getCurrency()->getCode());

        $this->assertEquals('PLN', MoneyFactory::fromFloat(1000.30)->getCurrency()->getCode());
        $this->assertEquals('USD', MoneyFactory::fromFloat(1000.30, 'USD')->getCurrency()->getCode());
    }

    /**
     * @test
     * @return void
     */
    public function it_pretty_prints_value(): void
    {
        $this->assertEquals('£ 10', MoneyFactory::prettyPrint(Money::GBP(1000)));
        $this->assertEquals('£ 10.30', MoneyFactory::prettyPrint(Money::GBP(1030)));
        $this->assertEquals('£ 10.33', MoneyFactory::prettyPrint(Money::GBP(1033)));

        MoneyFactory::$prettyPrintFormat = MoneyFactory::AMOUNT_SPACE_CURRENCY_FORMAT;

        $this->assertEquals('10.33 PLN', MoneyFactory::prettyPrint(Money::PLN(1033)));
        $this->assertEquals('10.30 £', MoneyFactory::prettyPrint(Money::GBP(1030)));

        MoneyFactory::$prettyPrintFormat = MoneyFactory::CURRENCY_AMOUNT_FORMAT;

        $this->assertEquals('£10.30', MoneyFactory::prettyPrint(Money::GBP(1030)));
        $this->assertEquals('$10.30', MoneyFactory::prettyPrint(Money::USD(1030)));
    }

    /**
     * @test
     * @return void
     */
    public function it_pretty_prints_negative_value(): void
    {
        MoneyFactory::$prettyPrintFormat = MoneyFactory::CURRENCY_AMOUNT_FORMAT;

        $this->assertEquals('£10', MoneyFactory::prettyPrint(Money::GBP(1000)));
        $this->assertEquals('- £10', MoneyFactory::prettyPrint(Money::GBP(-1000)));
    }

    /**
     * @test
     * @return void
     */
    public function it_json_serializes_value(): void
    {
        MoneyFactory::$prettyPrintFormat = MoneyFactory::CURRENCY_SPACE_AMOUNT_FORMAT;

        $expectedResult = [
            'amount' => '1000',
            'currency' => new Currency('GBP'),
            'formatted' => '£ 10',
            'float' => 10.0
        ];

        $this->assertEquals($expectedResult, MoneyFactory::jsonSerialize(Money::GBP(1000)));
    }
}
