<?php

declare(strict_types=1);

namespace DeMarketplace\Support;

class HMAC
{
    const HASHING_ALGO = 'sha256';

    /**
     * @var string
     */
    protected string $key;

    /**
     * @var null|string
     */
    protected ?string $hashedData = null;

    /**
     * @param string $key
     * @param null|string $hashedData
     */
    public function __construct(string $key, ?string $hashedData = null)
    {
        $this->key = $key;
        $this->hashedData = $hashedData;
    }

    /**
     * @param string $key
     * @param string $stringToHash
     * @return HMAC
     */
    public static function create(string $key, string $stringToHash): HMAC
    {
        return (new static($key))->hash($stringToHash);
    }

    /**
     * @param string $key
     * @param string $hashedData
     * @return HMAC
     */
    public static function createFromHash(string $key, string $hashedData): HMAC
    {
        return new static($key, $hashedData);
    }

    /**
     * @param string $stringToHash
     * @return HMAC
     */
    public function hash(string $stringToHash): HMAC
    {
        $this->hashedData = hash_hmac(self::HASHING_ALGO, $stringToHash, $this->key);

        return $this;
    }

    /**
     * @param HMAC $hashToCompare
     * @return bool
     */
    public function isEqual(HMAC $hashToCompare): bool
    {
        return hash_equals($this->hashedData, $hashToCompare->hashedData);
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return $this->hashedData;
    }
}
